#' @title Bayes factors for Bayesian exploratory and confirmatory hypothesis
#' testing
#' @description The \code{BF} function can be used for hypothesis testing and
#'  model
#' selection using the Bayes factor. By default exploratory hypothesis tests are
#' performed of whether each model parameter equals zero, is negative, or is
#'  positive.
#' Confirmatory hypothesis tests can be executed by specifying hypotheses with
#' equality and/or order constraints on the parameters of interest.
#'
#' @param x An R object containing the outcome of a statistical analysis.
#' An R object containing the outcome of a statistical analysis. Currently, the
#' following objects can be processed: t_test(), bartlett_test(), lm(), aov(),
#' manova(), cor_test(), lmer() (only for testing random intercep variances),
#' glm(), coxph(), survreg(), polr(), zeroinfl(), rma(), and named vector objects.
#' See README for elaborations.
#' @param hypothesis A character string containing the informative hypotheses to
#' evaluate. The default is NULL, which will result in standard exploratory testing
#' under the model \code{x}.
#' @param prior A vector specifying the prior probabilities of the hypotheses.
#' The default is NULL which will specify equal prior probabilities.
#' @param complement a logical specifying whether the complement should be added
#' to the tested hypothesis under \code{hypothesis}.
#' @param ... Parameters passed to and from other functions.
#' @return The output is an object of class \code{BF}. The object has elements:
#' BFtu_exploratory, PHP_exploratory, BFtu_confirmatory, PHP_confirmatory,
#' BFmatrix_confirmatory, BFtable_confirmatory, PHP_interaction, prior,
#' hypotheses, estimates, model, call.
#' @details The function requires a fitted modeling object. Current analyses
#' that are supported: \code{\link[bain]{t_test}},
#' \code{\link[BFpack]{bartlett_test}},
#' \code{\link[stats]{aov}}, \code{\link[stats]{manova}},
#' \code{\link[stats]{lm}}, \code{mlm},
#' \code{\link[stats]{glm}}, \code{\link[polycor]{hetcor}},
#' \code{\link[lme4]{lmer}}, \code{\link[survival]{coxph}},
#' \code{\link[survival]{survreg}},
#' \code{\link[pscl]{zeroinfl}}, \code{\link[metafor]{rma}} and \code{\link[MASS]{polr}}.
#'
#' For testing parameters from the results of t_test(), lm(), aov(),
#' manova(), and bartlett_test(), hypothesis testing is done using
#' adjusted fractional Bayes factors are computed.
#' For testing measures of association (e.g., correlations) via \code{cor_test()},
#' Bayes factors are computed using joint uniform priors under the correlation
#' matrices. For testing intraclass correlations (random intercept variances) via
#' \code{lmer()}, Bayes factors are computed using uniform priors for the intraclass
#' correlations. For all other tests, an approximate Bayes factors are computed using
#' Gaussian approximations, similar as a classical Wald test.
#'
#' @references Mulder, J., D.R. Williams, Gu, X., A. Tomarken,
#' F. Böing-Messing, J.A.O.C. Olsson-Collentine, Marlyne Meyerink, J. Menke,
#' J.-P. Fox, Y. Rosseel, E.J. Wagenmakers, H. Hoijtink., and van Lissa, C.
#' (submitted). BFpack: Flexible Bayes Factor Testing of Scientific Theories
#' in R.
#' @examples
#' \dontshow{
#' # EXAMPLE 1. One-sample t test
#' ttest1 <- t_test(therapeutic, mu = 5)
#' print(ttest1)
#' # confirmatory Bayesian one sample t test
#' BF1 <- BF(ttest1, hypothesis = "mu = 5")
#' summary(BF1)
#' # exploratory Bayesian one sample t test
#' BF(ttest1)
#'
#' # EXAMPLE 2. ANOVA
#' aov1 <- aov(price ~ anchor * motivation,data = tvprices)
#' BF1 <- BF(aov1, hypothesis = "anchorrounded = motivationlow;
#'                               anchorrounded < motivationlow")
#' summary(BF1)
#'
#' # EXAMPLE 3. Logistic regression
#' fit <- glm(sent ~ ztrust + zfWHR + zAfro + glasses + attract + maturity +
#'    tattoos, family = binomial(), data = wilson)
#' BF1 <- BF(fit, hypothesis = "ztrust > zfWHR > 0;
#'                              ztrust > 0 & zfWHR = 0")
#' summary(BF1)
#'
#' # EXAMPLE 4. Correlation analysis
#' set.seed(123)
#' cor1 <- cor_test(memory[1:20,1:3])
#' BF1 <- BF(cor1)
#' summary(BF1)
#' BF2 <- BF(cor1, hypothesis = "Wmn_with_Im > Wmn_with_Del > 0;
#'                               Wmn_with_Im = Wmn_with_Del = 0")
#' summary(BF2)
#' }
#' \donttest{
#' # EXAMPLE 1. One-sample t test
#' ttest1 <- bain::t_test(therapeutic, mu = 5)
#' print(ttest1)
#' # confirmatory Bayesian one sample t test
#' BF1 <- BF(ttest1, hypothesis = "mu = 5")
#' summary(BF1)
#' # exploratory Bayesian one sample t test
#' BF(ttest1)
#'
#' # EXAMPLE 2. ANOVA
#' aov1 <- aov(price ~ anchor * motivation, data = tvprices)
#' # check the names of the model parameters
#' names(aov1$coefficients)
#' BF1 <- BF(aov1, hypothesis = "anchorrounded = motivationlow;
#'                               anchorrounded < motivationlow;
#'                               anchorrounded > motivationlow")
#' summary(BF1)
#'
#' # EXAMPLE 3. Logistic regression
#' fit <- glm(sent ~ ztrust + zfWHR + zAfro + glasses + attract + maturity +
#'    tattoos, family = binomial(), data = wilson)
#' BF1 <- BF(fit, hypothesis = "ztrust > (zfWHR, zAfro) > 0;
#'                              ztrust > 0 & zfWHR=zAfro= 0")
#' summary(BF1)
#'
#' # EXAMPLE 4. Correlation analysis
#' set.seed(123)
#' cor1 <- cor_test(memory[1:20,1:3])
#' BF1 <- BF(cor1)
#' summary(BF1)
#' BF2 <- BF(cor1, hypothesis = "Wmn_with_Im > Wmn_with_Del > 0;
#'                               Wmn_with_Im = Wmn_with_Del = 0")
#' summary(BF2)
#' }
#' @rdname BF
#' @export
#' @useDynLib BFpack, .registration = TRUE
#'
BF <- function(x, hypothesis, prior, complement, ...) {
  UseMethod("BF", x)
}
