##############################################################################
#
# bard plan refinement
#
# 
#
# The public refineXXXplan methods all accept the following arguments
# (others may also be accepted):
#
#   plan - the starting plan
#   score.fun - the score function to be used,
#       this should accept the following arguments:
#          - plan
#          - 
#          accept return a single score, or a vector of N scores (one for
#          each district) which will 
#   displaycount - whether to provide a demo plot (slow)
#   historysize - size of score history to keep
#   dynamicscoring - whether to track changes and use incremental scoring
#   usecluster - whether it is permissible to use bard cluster 
#               Note: only refineGenoudPlan supports this currently
#   tracelevel - diagnostic traces. Zero means no tracing. 1 means
#       basic progress diagnostics, 2 is more detailed, 3 is verbose 
#
# Internally, a score wrapper is used to support change tracking,
# discreteness constraints, visualization.
#
#############################################################################

#################################
#
# refineGreedyPlan, refineTabuPlan
#
# This uses hill-climbing and tabu search to refine plans. 
#
# Greedy  will always accept the best candidate in the iteration. 
#
# Tabu
#     tabu search algorithm
#     Tabu list contains list of previous used district exchanges
#     Aspiration criterion is beating the best score
#     A sample of the possible exchanges are checked each time
#
# The neighborhood is defined as the set of plans generated by moving
# a single block from one district to a district that borders it.
#
#
# Arguments
#       Arguments as above. And note:
#        -  usecluster is ignored
#                          
# Returns
#    - new bard plan
#
# SEE R HELP FILE  FOR FULL DETAILS
# 
##################################

refineGreedyPlan <- function(plan, score.fun, displaycount=NULL, historysize=0, dynamicscoring=FALSE,   tracelevel=1, checkpointCount=0, resume=FALSE ) {
    retval <- refineGreedyPlanPlus(plan=plan,score.fun=score.fun,displaycount=displaycount,historysize=historysize,
     dynamicscoring=dynamicscoring,tracelevel=tracelevel,tabusize=0, checkpointCount=checkpointCount, resume=resume)
    return(retval)     
}

refineTabuPlan <- function(plan, score.fun, displaycount=NULL, historysize=0, dynamicscoring=FALSE,   tracelevel=1,tabusize=100, tabusample=500, checkpointCount=0, resume=FALSE) {
    retval <- refineGreedyPlanPlus(plan=plan,score.fun=score.fun,displaycount=displaycount,historysize=historysize,
    dynamicscoring=dynamicscoring,tracelevel=tracelevel,tabusize=tabusize, checkpointCount=checkpointCount, resume=resume)
    return(retval)     
}


#################################
#
# refineNelderPlan
#
# This uses nelder-mead to refine plans. It will always accept the best
# candidate in the iteration. 
#
# The neighborhood is defined as the set of plans generated by moving
# a single block from one district to a district that borders it.
#
#
# Arguments
#       Arguments as above. And note:
#        -  usecluster is ignored
#                          
# Returns
#    - new bard plan
#
# SEE R HELP FILE  FOR FULL DETAILS
# 
##################################


refineNelderPlan <- function(plan, score.fun, displaycount=NULL,  maxit=NULL,
 historysize=0, dynamicscoring=FALSE,   tracelevel=1 ) {
  
  control<-list()                                                        
  control$fnscale<-1
  if (tracelevel>1) {
    control$trace=9
  }
  if (is.null(maxit)) { 
    control$maxit<-10*length(plan)
  } else {
    control$maxit <- maxit
  }

                 
  # get plan stuff
  ndists<-attr(plan,"ndists")
  basemap<-basem(plan)

  
  nelderScoreFun <-
        scoreWrapper(score.fun,plan,displaycount,historysize,dynamicscoring,boundCheck=TRUE, tracelevel=tracelevel)

  nelderResult <- optim(plan,nelderScoreFun, method="Nelder-Mead", control=control)
  retval <- convertPar2Plan(nelderResult$par,plan)
  
  if (!is.null(displaycount)) {
    # wrap this in try, in case the window gets closed manually in mid run
    try(plot(retval)) 
  }
  return(retval)     
}

#################################
#
# refineGenoudPlan
#
# This uses an auto-configured genoud() optimization to refine plans.
#
# The neighborhood is defined as the set of plans generated by moving
# a single block from one district to a district that borders it.
#
#
# Arguments
#       Arguments as above. And note:
#        -  usecluster is used if available
#                          
# Returns
#    - new bard plan
#
# SEE R HELP FILE  FOR FULL DETAILS
# 
##################################

refineGenoudPlan <- function(plan, score.fun, displaycount=NULL,  historysize=0, dynamicscoring=FALSE ,   usecluster = TRUE, tracelevel=1 ) {
              
  if(!mrequire("rgenoud", quietly = TRUE, warn.conflicts=FALSE)) {
    stop("Sorry, this requires the rgenoud package. Please install it.")
  }
              
  if (dynamicscoring) {
      warning("Incremental score functions are not likely to increase performance with genetic algorithms")
  }          

  if (tracelevel>1) {
    print.level<-2
  } else {
      print.level<-0
  }
  if (usecluster && !is.null(setBardCluster())) {
    genCluster <- setBardCluster()
    memMat <- TRUE
    displaycount<-0
  } else {
    genCluster <- FALSE
    memMat <- FALSE
  }                                
               
  # get plan stuff
  ndists<-attr(plan,"ndists")
    basemap<-basem(plan)
  
  
  genoudScoreFun <- 
    scoreWrapper(score.fun,plan,displaycount,historysize,
      dynamicscoring,boundCheck=F, tracelevel=tracelevel)


  # adjust genoud params to plan
  # P3 = 0 because boundary mutation doesn't make sense for district ID's
  # P7 = 0 -- whole non-uniform mutations does not make sesne for this case
  # P8, P4 , might make sense , needs experi mentation
  
  max(pop.size<-sqrt(length(plan))*100,10000)
  max.generations<-length(plan)
  wait.generations<-sqrt(length(plan))
  
  
  genoudResult <- genoud( fn = genoudScoreFun,  nvars=length(plan),  
    pop.size=pop.size, hard.generation.limit=FALSE,
    max.generations=max.generations,
    wait.generation=wait.generations,
    MemoryMatrix=memMat, cluster=genCluster, 
    P3=0, P7=0,
    P4=0, P8=0,
    Domains=t(replicate(length(plan),c(1,ndists))),
    solution.tolerance=0.001, boundary.enforcement=2,
    print.level=print.level,
    data.type.int=T,  starting.values=c(as.double(plan)) )
  retval <- convertPar2Plan(genoudResult$par,plan)
  
  if (!is.null(displaycount)) {
    # wrap this in try, in case the window gets closed manually in mid run
    try(plot(retval)) 
  }
  return(retval)     
}

#################################
#
# refineAnnealPlan
#
# This uses simulated to refine plans. 
#
# The temperature and other parameters are auto-tuned.
#
# The generation function samples from a neighborhood of
# -  single moves of blocks from one district to an adjoining district
# - switches of pairs of adjoining blocks in different districts 
#
# The neighborhood is defined as the set of plans generated by moving
# a single block from one district to a district that borders it.
#
#
# Arguments
#       Arguments as above. And note:
#        -  usecluster is ignored
#                          
# Returns
#    - new bard plan
#
# SEE R HELP FILE  FOR FULL DETAILS
# 
##################################

refineAnnealPlan <- function(plan, score.fun, 
     displaycount=NULL, 
     historysize=0, dynamicscoring=FALSE,
     tracelevel=1, greedyFinish=FALSE ) {
               
  
  # get plan stuff
  ndists<-attr(plan,"ndists")
    basemap<-basem(plan)
  
  
  # control param adjust
  control<-list()
  control$fnscale<-1
  if (tracelevel>1) {
    control$trace<-9
  }

  control$temp<-11   # 'These go to 11.' -- Nigel Tufnel
  control$tmax<-min(500,max(10,ceiling(length(plan)/10)))
  control$maxit<-2000*control$tmax
  
  if (tracelevel>0) {
    print(paste("Starting Annealing for", control$maxit,"iterations"))
  }
  
  canGen <- function(par) {
    legal <- convertPar2Plan(par,plan,boundCheck=F)
    candidateBlocks <- locallyExchangeableBlocks(legal)
    if (length(candidateBlocks)==0) {
        print("Annealing: out of candidates.")     # DEBUGGING ONLY
        return(par)
    }
    newpar<-par
    ex1<-sample(candidateBlocks,1)
    
    nex <- neighbors(basemap$nb,ex1)
    nex2 <- nex[which(par[nex]!=ex1)]
    ex2 <- sample(nex2,1)
    newpar[ex1]<-newpar[ex2]
    if (runif(1)>.5) {
      newpar[ex2]<-par[ex1]
    }
    return(newpar)
  }
                                                            
  annealScoreFun <-  scoreWrapper(score.fun,plan,displaycount,historysize,dynamicscoring,
        boundCheck=FALSE, tracelevel=tracelevel)
  
  annealResult <- optim(plan,annealScoreFun, method="SANN", 
      gr=canGen, control=control)
  retval <- convertPar2Plan(annealResult$par,plan,boundCheck=F)

  if (greedyFinish) {
    print("Anneal done, finishing up with hill climbing...")
    displaycount<-ceiling(displaycount/100)
    retval <-refineGreedyPlan(retval, score.fun, displaycount=displaycount, 
       dynamicscoring=dynamicscoring,  tracelevel=tracelevel)
  }
  

  if (!is.null(displaycount)) {
    plot(retval)
  }
  
  return(retval)     
}



##############################################################################
#           INTERNAL MODULE FUNCTIONS -- DO NOT EXPORT
###############################################################################


#
# Workhorse for Greedy and Tabu
#


refineGreedyPlanPlus <- function(plan, score.fun, displaycount=NULL, historysize=0,
	tabusize =0, dynamicscoring=FALSE,   tracelevel=1,tabusample=500,
	checkpointCount=0, resume=FALSE) {
                                                          
  if (resume) {
      checkpoint.env<-get("BardCheckPoint",envir=.GlobalEnv)
      lslist=setdiff(ls(checkpoint.env),c("checkpoint.env","resume","checkpointCount"))
      for (enitem in lslist) {
	 assign(enitem,get(enitem,envir=checkpoint.env))
      }   
  }
  
  if (checkpointCount>0 && !resume) {
        checkpoint.env<-new.env()
  	assign("BardCheckPoint",checkpoint.env,envir=.GlobalEnv)
  }
  
  
  if (!resume) {
   # no score history -- greedy plan generation doesn't revisit plans
   
   if (tabusize==0) {
	historysize<-0
   } else {
	tabulist <- matrix(nrow=tabusize,ncol=2)
   }
            
   # get plan stuff
   ndists<-attr(plan,"ndists")
   basemap<-basem(plan)
   nb<-basemap$nb
  
   # tracking vars
   bestScore<-prevscore<-score.fun(plan)
   mitercount<-0
   lastplan<-plan
   lastplotted<-plan
  
   if (!is.null(displaycount)) {
      try(plot(plan))
    }
  
   # helper functions
   candpairs<-function(i) {
     candres<-as.matrix(expand.grid(i,neighbors(nb,i)))
	 # later for switches eliminate duplicates
	 # candres<-t(apply(candres,1,function(x)if(x[1]>x[2]){c(x[1],x[2])}else{c(x[2],x[1])}))
	 # candres<-unique(candres)
	 return(candres)
   }
  
   
   # tricky -- displaycount in scorewrapper set to NULL, since we display inside
   # the repeat loop, adjust tracelevel
   
   greedyScore <-
        scoreWrapper(score.fun,plan,
        displaycount=NULL,historysize=historysize,dynamicscoring=dynamicscoring,
        boundCheck=FALSE, tracelevel=max(0,tracelevel-1))
   
   # scores a single exchange from the current plan
   scoreMove<-function(spair,curplan,returnScore=TRUE,
    lastscore=NULL ) {
    home<-spair[1]
    newid<-spair[2]
    testplan<-curplan
    testplan[newid]<-plan[home]
    changelist<-matrix(ncol=2, c(newid,curplan[newid]))
    
    if (returnScore) {
      return(greedyScore(testplan,changelist=changelist,lastscore=lastscore))
    } else {
      return(testplan)
    }
   }

   maxiter <- length(plan)
   iterSinceImproved<-0
   mitercount<-0
   bestPar<-NULL
  } # end if resume
  
  repeat {
     mitercount <- mitercount +1
	 iterSinceImproved<-iterSinceImproved +1
	 
	 # new candidate  is best of single one way moves
     lb<-locallyExchangeableBlocks(lastplan) 
	 ltmp<-lapply(lb,candpairs)
     lpairs<-cbind( unlist(lapply(ltmp,function(x)x[,1])),
         unlist(lapply(ltmp,function(x)x[,2]))) 
	
	 # Later -- for 2 d exchanges dedup
	 #lpairs<-t(apply(lpairs,1,function(x)if(x[1]>x[2]){c(x[1],x[2])}else{c(x[2],x[1])}))
	 #lpairs<-unique(lpairs)
	 
	 # single switch only
	 dupid<-duplicated(cbind(lpairs[,1],lastplan[lpairs[,2]]))
	 lpairs<-lpairs[!dupid,,drop=F]
	 
     if (tracelevel) {
        print(paste("MAJOR ITERATION ",mitercount,": total ",dim(lpairs)[1],"candidates",Sys.time())) 
        flush.console()
     }   
	if (tabusize>0) {
			samplesize <- min(dim(lpairs)[1],tabusample)
			ind<-sample(dim(lpairs)[1],samplesize)
			lpairs<-lpairs[ind,,drop=F]
				if (tracelevel) {
					print(paste("Evaluating ",dim(lpairs)[1],"candidates",Sys.time())) 
					flush.console()
				} 
	 }
  
     if (dynamicscoring) {
       lpairsScores<- apply(lpairs,1,
        function(x)scoreMove(x,curplan=lastplan,lastscore=prevscore))
     } else {
        lpairsScores<- apply(lpairs,1,function(x)scoreMove(x,curplan=lastplan))
     }
      
     bestswitch<-which.min(lpairsScores)
     candidatePar<-scoreMove(lpairs[bestswitch,],curplan=lastplan,returnScore=F)
     
	 curscore <- lpairsScores[bestswitch]
     if (sum(curscore)>=sum(bestScore)) {
		# non-improvement -- what to do...

	  if (tabusize==0) {
		#browser()
		break
		# GREEDY, local opt
		# finished ...
	  } else if (iterSinceImproved > maxiter) {
		break
		# TABU max iterations
		# finished...
	  } else if (any(apply(na.omit(tabulist),1,function(x)all(x==lpairs[bestswitch,])))) {
		# TABU current solution on tabu list
		
		if (tracelevel) {
			print(paste("Current solution is TABUed, iterSinceImproved:", iterSinceImproved)) 
			flush.console()
		}  
		
		# exclude other tabu solutions
		tmpm<- rbind(lpairs,unique(na.omit(tabulist)))
		dups<-duplicated(tmpm,fromLast=TRUE)[1:dim(lpairs)[1]]
		
		# any solutions left? 
		if (sum(dups)>0) {
			legalscores<-lpairsScores[!dups]
			legalpairs<-lpairs[!dups,,drop=F]
			bestlegal<-which.min(legalscores)
			candidatePar<-scoreMove(legalpairs[bestlegal,,drop=F],curplan=lastplan,returnScore=F)
			tabulist [mitercount %% tabusize,] <- legalpairs[bestlegal,,drop=F]

		} else {
			if (tracelevel) {
				print(paste("All tabued")) 
				flush.console()
			}  
		}
		
	  }
     } else {
		# improvement -- meets greedy and tabu aspiration criteria
		iterSinceImproved=0
		bestScore<-curscore
		bestPar <- candidatePar
		if (tabusize>0) {
			tabulist [mitercount %% tabusize,] <- lpairs[bestswitch,]
		}
	 }


     # eval score
     score<-score.fun(candidatePar)
     
     # post tracking
     prevscore<-score
     lastplan<-candidatePar
     
     if (tracelevel) {
        print(paste("MAJOR ITERATION:  Best Score",bestScore, "current score",sum(score),"iteration",mitercount,Sys.time())) 
        flush.console()
     }           
     

     if (!is.null(displaycount)) {
      
        if ((mitercount %% displaycount) ==0) {
           updatePlot(candidatePar,lastplotted,score,mitercount)
           lastplotted<-candidatePar  
        }
      }
     if (checkpointCount>0 && (mitercount %% checkpointCount ==0)) {
       
       	for (enitem in ls()) {
		assign(enitem,get(enitem),envir=checkpoint.env)
	}
     
     }
  }
  
  retval<-bestPar
  if (!is.null(displaycount)) {
    # wrap this in try, in case the window gets closed manually in mid run
    try(plot(retval)) 
  }
  return(retval)     
}



#  ScoreWrapper Functions
# 
#  The workhorse is scoreWrapper, updatePlot() and convertPar2Plan() are support 
# functions called by  scoreWrapper
#



#################################
#
# scoreWrapper
#
# This is a workhorse function used by most of the refineXXXplan functions to
# prepare a bard score function for use with a general purpose optimizer such 
# as optim()
#
# It uses a closure to store tracking variables for iteration counts,
# score histores, etc. It also performs plot updating for demos,
# boundschecking for optimizers that work in continuous space.
#
#
# Arguments
#    See the description of refineXXXplan above
#   
#   Additionally:
#
#  - FUN - original score function (score.fun)
#  - boundCheck - whether to perform boundschecking on input parameter   
#                          
# Returns
#    - new score function
# 
#################################
  
scoreWrapper<-function(FUN,plan,
  displaycount,historysize,dynamicscoring,
  boundCheck=TRUE,tracelevel=0)  {
    #
    # Setup tracking variables in closure
    #
    
  # Tricky Stuff -- this contains lexical vars updated by nested functions
  #                 when the nested score function is called by the optimizers
  #                 this allows us to update plots, keep track of changes
  #                 for incremental scoring, and keep a score history
  #
  # itercount - iteration count for plot updates
  # lastplan - last plan seen for purposes of generating a list of changes
  # history - history of previous function evaluations  
     DEBUG<-FALSE
    
    # change tracking
    lastplotted<-integer(length(plan))
    lastplan<-integer(length(plan))
    itercount<-0
    scorehistory <- vector(historysize,mode="list")
    changelist<-NULL
    savedscore<-NULL
    
    # check dynamic change support
    if (dynamicscoring) {
      funformals<-names(formals(FUN))
      if (!("..." %in% funformals) 
      && !(all(c("lastscore","changelist") %in% funformals) ))       {
        warning("score function does not appear to support dynamic scoring")
        dynamicscoring<-FALSE
      }
      savedscore<-FUN(plan)
      lastplan<-plan
    }
    
    # plotting for demos
    if (!is.null(displaycount)) {
      try(plot(plan))
    }
    
    # build wrapped score function
    wrappedScoreFun <- function(candidatePar,lastscore=NULL,changelist=NULL) {
      # prepare for Function run
     
      itercount<<-itercount+1
      
      # check history
      score<-NULL
      if (historysize>0) {
        plandig <- digest(candidatePar)
        score <- scorehistory[[plandig]]
        if (DEBUG) {
          print(paste("history",plandig,score))
          flush.console()
        }
      }
      if (is.null(score)) {
        candidatePar<-convertPar2Plan(candidatePar,plan,boundCheck)
        # track blocks that changed
        if (dynamicscoring && is.null(changelist)) {
            changelist<-which(lastplan!=candidatePar)
            if (length(changelist)>0) {
              changelist<-cbind(changelist,lastplan[changelist])
            }
            if (DEBUG) {
              print(paste("dynamic",savedscore,"changes",paste(collapse=" ",changelist)))
                flush.console()
            }
        } 
      
        # run score function
       if (dynamicscoring) {
          if (is.null(lastscore)) {
            lastscore<-savedscore
          }
      
          if (length(changelist)==0) {
            score<-lastscore
          } else {
            score<-FUN(candidatePar,lastscore=lastscore,changelist=changelist)
            
          }
        } else {
          score<-FUN(candidatePar)
        }
        #update tracking
        if (historysize>0) {
          names(scorehistory)[itercount%%historysize]<<-plandig
          scorehistory[itercount%%historysize]<<-score
        }
      }
    
      if ((tracelevel>1) || ((tracelevel>0) && ((itercount %% 100) ==0))) {
        print(paste("Score",paste(score,collapse=" "),"iteration",itercount,Sys.time())) 
        flush.console()
      }
      if (!is.null(displaycount)) {
        if ((itercount %% displaycount) ==0) {
            # (re)convert, in case the score was taken from a history list
            candidatePar<-convertPar2Plan(candidatePar,plan,boundCheck)
            
            updatePlot(candidatePar,lastplotted,score,itercount)
            lastplotted<<-candidatePar
        }
      }
      if (dynamicscoring) {
         lastplan<<-candidatePar
         savedscore<<-score
      }
      return(sum(score)) 
    }
    return(wrappedScoreFun)
  }

#################################
#
# updatePlot
#
# For demo's, updates a plot of the current score being evaluted
#
# Arguments
#
#  - plan  -  plan to be plotted
#  - oldplan - previously plotted plan, if known
#  - score - current plan score
#  - itercount - iteration count
#                          
# Returns
#    - NULL
# 
#################################

updatePlot<-function(plan,oldplan=NULL,score=0,itercount=0,refreshcount=500,skipplotcount=1,printit=T
, sleepTime=.05) {
  if (is.null(oldplan)) {
    changed<-1:length(plan)
  } else {
    changed <- which(plan!=oldplan)
  }

  if ((length(changed)>0) && (!itercount %% skipplotcount)) {
    
    if ((skipplotcount<2) && itercount %% refreshcount) {
      # WARNING: newPlot FALSE creates memory leak because of add flag 
      #          in plot.polylist
      res <- try(plot(plan,changed=changed,newPlot=FALSE),silent=TRUE)
       Sys.sleep(sleepTime)


    } else {
    
       # use try, in case window got closed manually
       #try(dev.off(), silent=TRUE)
       res <- try(plot(plan,newPlot=TRUE),silent=TRUE)
       Sys.sleep(sleepTime)
       
    }
    
    if (inherits(res,"try-error")) {
       try(plot(plan,newPlot=TRUE),silent=TRUE)
    }
    
    } else {
    }
  if (printit>0) {
      print(paste("Iterations:",itercount,"(",paste(score,collapse=" "),")*", Sys.time()) )
      flush.console()


  }

  return(NULL)
}

#################################
#
# convertPar2Plan
#
# Most optimization function return numeric character vectors.
# This puts them back into bardPlan form in order to evaluate the scores
#
# Arguments
#
#  - candidatePar  -  optimization vector
#  - plan - original plan
#  - boundCheck - for optimizers that do not respect discreteness and
#                 district ID contraints. Forces par to bounds
#  - itercount - iteration count
#                          
# Returns
#    - new bardplan
# 
#################################

convertPar2Plan <- function(candidatePar,plan,boundCheck=TRUE) {
      ndists<-attr(plan,"ndists")
      
      if (boundCheck) {
         candidatePar<-as.integer(pmin(pmax(1,round(candidatePar)),ndists))
      } else if (!is.integer(candidatePar)) {
         candidatePar<-as.integer(candidatePar) 
      }
        
      attributes(candidatePar)<-attributes(plan)
      return(candidatePar)
}

       
####################
# Misc functions
####################
                         


#################################
#
# locallyExchangeableBlocks
#
# Returns list of blocks in a plan that border blocks in 
# a differnt district
#
# Arguments
#
#  - plan  -  candidate plan
#                          
# Returns
#    - list of blockid's
# 
#################################

locallyExchangeableBlocks<- function(plan) {

  ndists <- attr(plan,"ndists")
  
  lebD<-function(plan,distid) {
    blocks<-which(plan==distid)
    if (length(blocks)==0) {
      return(NULL)
    }
    
    basemap<-basem(plan)
    candidates<- neighbors(basemap$nb,blocks)
    ret<-candidates[which(plan[candidates]!=distid)]
    return(ret)
  }
  
  retval<-
  unique(c(sapply(1:(ndists-1), function(x)lebD(plan,x)),recursive=T))
   
   return(retval)
   
}

####################
# Test functions - Used for development testing, not used in running production code.
####################


annealTempDecay<-function(temp,tmax,maxit) {

    curtemp <-  temp / log(((maxit-1) %/% tmax)*tmax + exp(1))
    return(curtemp)
}
