hist <- function(x, ...) UseMethod("hist")

hist.default <-
    function (x, breaks, freq= NULL, probability = !freq, include.lowest= TRUE,
              right= TRUE, col = NULL, border = par("fg"),
              main = paste("Histogram of" , xname),
              xlim = range(breaks), ylim = NULL,
              xlab = xname, ylab,
              axes = TRUE, plot = TRUE, labels = FALSE, nclass = NULL, ...)
{
    if (!is.numeric(x))
        stop("`x' must be numeric")
    xname <- deparse(substitute(x))
    n <- length(x <- x[!is.na(x)])
    use.br <- !missing(breaks)
    if(use.br) {
        if(!missing(nclass))
            warning("`nclass' not used when `breaks' specified")
    }
    else if(!is.null(nclass) && length(nclass) == 1)
        breaks <- nclass
    use.br <- use.br && (nB <- length(breaks)) > 1
    if(use.br)
        breaks <- sort(breaks)
    else {                              # construct vector of breaks
        rx <- range(x)
        nnb <-
            if(missing(breaks)) 1 + log2(n)
            else {                      # breaks = `nclass'
                if (is.na(breaks) | breaks < 2)
                    stop("invalid number of breaks")
                breaks
            }
        breaks <- pretty (rx, n = nnb, min.n=1)

        nB <- length(breaks)
        if(nB <= 1) ##-- Impossible !
            stop(paste("hist.default: pretty() error, breaks=",format(breaks)))
    }

    ## Do this *before* adding fuzz or logic breaks down...

    h <- diff(breaks)
    equidist <- !use.br || diff(range(h)) < 1e-7 * mean(h)
    if (!use.br && any(h <= 0))
        stop("not strictly increasing `breaks'.")
    if (is.null(freq)) {
        freq <- if(!missing(probability)) !as.logical(probability) else equidist
    } else if(!missing(probability) && any(probability == freq))
        stop("`probability' is an alias for `!freq', however they differ.")

    ## Fuzz to handle cases where points are "effectively on"
    ## the boundaries
    diddle <- 1e-7 * max(abs(range(breaks)))
    fuzz <- if(right)
	c(if(include.lowest)-diddle else diddle,
	    rep(diddle, length(breaks) - 1))
    else
	c(rep(-diddle, length(breaks) - 1),
	    if(include.lowest) diddle else -diddle)

    breaks <- breaks + fuzz
    h <- diff(breaks)

    storage.mode(x) <- "double"
    storage.mode(breaks) <- "double"
    ## With the fuzz adjustment above, the "right" and "include"
    ## arguments are really irrelevant
    counts <- .C("bincount",
                 x,
                 n,
                 breaks,
                 nB,
                 counts = integer(nB - 1),
                 right = as.logical(right),
                 include= as.logical(include.lowest),
                 NAOK = FALSE, DUP = FALSE, PACKAGE = "base") $counts
    if (any(counts < 0))
        stop("negative `counts'. Internal Error in C-code for \"bincount\"")
    if (sum(counts) < n)
        stop("some `x' not counted; maybe `breaks' do not span range of `x'")
    density <- counts/(n*h)
    mids <- 0.5 * (breaks[-1] + breaks[-nB])
    r <- structure(list(breaks = breaks, counts = counts,
                        intensities = density,
			density = density, mids = mids,
                        xname = xname, equidist = equidist),
                   class="histogram")
    if (plot) {
##-         if(missing(ylim))
##-             y <- if (freq) .Alias(counts) else .Alias(density)
        plot(r, freq = freq, col = col, border = border,
             main = main, xlim = xlim, ylim = ylim, xlab = xlab, ylab = ylab,
             axes = axes, labels = labels, ...)
        invisible(r)
    }
    else r
}

plot.histogram <-
    function (x, freq = equidist, col = NULL, border = par("fg"), lty = NULL,
              main = paste("Histogram of", x$xname),
              xlim = range(x$breaks), ylim = NULL,
              xlab = x$xname, ylab,
              axes = TRUE, labels = FALSE, add = FALSE, ...)
{
    equidist <-
        if(is.logical(x$equidist)) x$equidist
        else { h <- diff(x$breaks) ; diff(range(h)) < 1e-7 * mean(h) }
    if(freq && !equidist)
        warning("the AREAS in the plot are wrong -- rather use `freq=FALSE'!")

    y <- if (freq) x$counts else x$density
    nB <- length(x$breaks)
    if(is.null(y) || 0 == nB) stop("`x' is wrongly structured")

    if(!add) {
        if(is.null(ylim))
            ylim <- range(y, 0)
        if (missing(ylab))
            ylab <- if (!freq) "Density" else "Frequency"
        plot.new()
        plot.window(xlim, ylim, "")     #-> ylim's default from 'y'
        title(main = main, xlab = xlab, ylab = ylab, ...)
        if(axes) {
            axis(1, ...)
            axis(2, ...)
        }
    }
    rect(x$breaks[-nB], 0, x$breaks[-1], y,
         col = col, border = border, lty = lty)
    if((logl <- is.logical(labels) && labels) || is.character(labels))
        text(x$mids, y,
             labels = if(logl) {
                 if(freq) x$counts else round(x$density,3)
             } else labels,
             adj = c(0.5, -0.5))
    invisible()
}

lines.histogram <- function(x, ...) plot.histogram(x, ..., add = TRUE)
